function cds = impute_missing_cds_data(cds, max_lag, business_days)
% --------------------------------------------------------------------------------------------------
% Impute missing CDS spreads with most recent spreads, but only if less than 'max_lag' days in the
% past.
% --------------------------------------------------------------------------------------------------
% cds                      ... credit default swap structure
% max_lag                  ... maximum number of days that imputed data can be old
% business_days            ... days for which data would be desireable
% --------------------------------------------------------------------------------------------------
% sample call:
%{
    test = CDX_NA_IG2.portfolio(4);
    impute_missing_cds_data(test, 7, CDX_NA_IG2.dates{1})
%}
% --------------------------------------------------------------------------------------------------

% Do imputation for each maturity saperately
for i=1:length(cds.dates)
    % Determine missing dates
    dates = cds.dates{i};
    available_dates = is_member_sorted_c(business_days, dates);
    missing_dates = find(~logical(available_dates))
    if (numel(missing_dates) == 0)
        continue;
    end
    
    % For each missing date, determine last available date
    last_available_pos = zeros(size(missing_dates));
    for j=1:length(missing_dates)
        tmp = find(dates < missing_dates(j), 1, 'last');
        if (~isempty(tmp))
            last_available_pos(j) = tmp;
        else
            last_available_pos(j) = -1;
        end
    end
    missing_dates = missing_dates(last_available_pos > 0);
    last_available_pos = last_available_pos(last_available_pos > 0);
    last_available_date = dates(last_available_pos);
    
    % Filter out dates, where last available data more than 'max_lag' days in the past
    for j=length(missing_dates):(-1):1
        day_diff = date_diff_years(last_available_date(j), missing_dates(j)) * 360;
        if (day_diff > max_lag)
            missing_dates = [missing_dates(1:(j-1)); missing_dates((j+1):end)];
            last_available_pos = [last_available_pos(1:(j-1)); last_available_pos((j+1):end)]; 
            last_available_date = [last_available_date(1:(j-1)); last_available_date((j+1):end)];
        end
    end
    
    % Impute data
    for j=length(missing_dates):(-1):1
        cds.dates{i} = [cds.dates{i}(1:last_available_pos); missing_dates(j); cds.dates{i}((last_available_pos+1):end)];
        cds.market_price{i} = [cds.market_price{i}(1:last_available_pos); cds.market_price{i}(last_available_pos); ...
                               cds.market_price{i}((last_available_pos+1):end)];                            
    end
end
